import { _decorator, Component, log, Node } from 'cc';
import { Popup } from './Popup';
import { Overlay } from './Overlay';
const { ccclass, property } = _decorator;

@ccclass('PopupManager')
export class PopupManager extends Component {
    private _popupsNode: Map<string, Node> = new Map();
    private _overlayNode: Node = null!;
    private _activePopupNode: Node | null = null;

    protected onLoad(): void {
        const popups = this.node.getComponentsInChildren(Popup);
        const overlay = this.node.getComponentInChildren(Overlay);

        if (popups?.length > 0 && overlay) {
            this._setPopups(popups);
            this._setOverlay(overlay);
        }
    }

    protected onDestroy(): void {
        this._popupsNode.forEach(popup => popup.off(Node.EventType.TOUCH_END, () => {}, this));
        this._overlayNode?.off(Node.EventType.TOUCH_END, this._hideAll, this);
    }

    public showPopup(uuid?: string) {
        this._showPopup(uuid);
    }

    private _showOverlay() {
        if (this._overlayNode && this._overlayNode.active === false) {
            this._overlayNode.active = true;
        }
    }

    private _hideOverlay() {
        if (this._overlayNode && this._overlayNode.active === true) {
            this._overlayNode.active = false;
        }
    }

    private _showPopup(uuid?: string) {
        if (!uuid) {
            log(`Невозможно открыть: ${uuid}`);
            return;
        }

        const popupNode = this._popupsNode.get(uuid);
        if (popupNode && popupNode.active === false) {
            this._hideAll();
            popupNode.active = true;
            this._showOverlay();
            this._activePopupNode = popupNode;
        } else {
            log(`Такого окна нет, или оно уже открыто.`);
        }
    }

    private _hidePopup(uuid?: string) {
        const popupNode = uuid ? this._popupsNode.get(uuid) : this._activePopupNode;
        if (popupNode && popupNode.active === true) {
            popupNode.active = false;
            this._hideOverlay();
            this._activePopupNode = null;
        }
    }

    private _hideAll() {
        this._hidePopup();
        this._hideOverlay();
    }

    private _setPopups(popups: Popup[]) {
        popups.forEach(popup => this._popupsNode.set(popup.node.uuid, popup.node));
        this._popupsNode.forEach(node => {
            node.active = false;
            node.on(Node.EventType.TOUCH_END, () => {}, this);
        });
    }

    private _setOverlay(overlay: Overlay) {
        this._overlayNode = overlay.node;
        this._overlayNode.active = false;

        this._overlayNode.on(Node.EventType.TOUCH_END, this._hideAll, this);
    }
}


